
#include "DAC.h"
#include "p33Fxxxx.h"
#include <string.h>

#define BUFFER_SIZE_WORDS 256
static signed short TxBufferA[BUFFER_SIZE_WORDS] __attribute__((space(dma)));
static signed short TxBufferB[BUFFER_SIZE_WORDS] __attribute__((space(dma)));

static unsigned short DACBuffer_writepos;
static volatile unsigned short DACBuffer_readpos;
volatile unsigned char DACBuffer_full, DACBuffer_empty;
static unsigned char DAC_CurrentBuffer;
static bool DAC_Paused;

signed short DACBuffer[256*DAC_BUFFER_PAGES];

void dac_init() {
	// RB1 = DACENA (output, active high)
	TRISBbits.TRISB1 = 0;
	LATBbits.LATB1 = 0;

	// RB4 = DIN (output, serial data to DAC)
	TRISBbits.TRISB4 = 0;
	LATBbits.LATB4 = 0;

	// RB7 = MCLK (output, master clock, 128 x Fs)
	TRISBbits.TRISB7 = 0;

	// RB9 = LRCIN (output, sample clock, Fs)
	TRISBbits.TRISB9 = 0;
	LATBbits.LATB9 = 0;

	// TMR2 used for generating master clock
	// TMR3 used for ADC, setting it up here so that it's synchronised
    PR2 = 4-1;           // Mclk frequency = 1/4 instruction clock
	T2CONbits.TON = 1;

	RPOR3bits.RP7R = 18; // RP7/RB7 (pin 16) = OC1
	OC1CONbits.OCM = 6;  // PWM
	OC1RS = 2;           // ~50% duty cycle

	// set up DCI for I2S
	RPOR2bits.RP4R = 13; // DCI CSDO
	RPOR4bits.RP9R = 15; // DCI COFS
	RPINR24bits.CSCKR = 10; // DCI CSCK
	DCICON1bits.COFSM = 1; // I2S
	DCICON1bits.CSCKE = 1; // required for I2S
	DCICON1bits.CSCKD = 1; // DCI clock is an input
	DCICON2bits.WS = 12-1;
	DCICON2bits.COFSG = 2-1;
	TSCON = 1|2;
	IFS3bits.DCIIF = 0;

	DMA1CONbits.DIR = 1;
	DMA1CONbits.MODE = 2;
	DMA1REQbits.IRQSEL = 0x3c;
	DMA1STA = __builtin_dmaoffset(TxBufferA);
	DMA1STB = __builtin_dmaoffset(TxBufferB);
	DMA1PAD = (volatile unsigned int) &TXBUF0;
	DMA1CNT = sizeof(TxBufferA)/2-1;
	IFS0bits.DMA1IF = 0;
	IEC0bits.DMA1IE = 1;
}

/*
	Crystal frequency = 8000000Hz, target operating frequency = 33868800Hz (44.1kHz x 192 x 4)
	Best match operating frequency = 33800000Hz (-0.203%) with N1=10, M=169, N2=2 (44.01kHz x 192 x 4)
	PLL input frequency is 800000Hz, output frequency is 135200000Hz
	PLLPRE=8 PLLFBD=167 PLLPOST=0
	
	Crystal frequency = 8000000Hz, target operating frequency = 36864000Hz (48kHz x 192 x 4)
	Best match operating frequency = 36857142.8571429Hz (-0.019%) with N1=7, M=129, N2=2 (47.99kHz x 192 x 4)
	PLL input frequency is 1142857.14285714Hz, output frequency is 147428571.428571Hz
	PLLPRE=5 PLLFBD=127 PLLPOST=0
*/

unsigned char dac_set_sample_rate(unsigned short rate) {
	unsigned char ret = 1;

    __builtin_write_OSCCONH(0x02); // switch to crystal oscillator
    __builtin_write_OSCCONL(0x01);

	if( rate == 44100 ) {
		CLKDIVbits.PLLPRE = 8;
  		PLLFBD = 167;
		CLKDIVbits.PLLPOST = 0;
		PR5 = 21125; // maintain timer1 at 200Hz
	} else if( rate == 48000 ) {
		CLKDIVbits.PLLPRE = 5;
  		PLLFBD = 127;
		CLKDIVbits.PLLPOST = 0;
		PR5 = 23026; // maintain timer1 at 200Hz
	} else {
		ret = 0;
	}

    __builtin_write_OSCCONH(0x03); // switch back to PLL
    __builtin_write_OSCCONL(0x01);
	
	return ret;
}

static const signed short DAC_zero[BUFFER_SIZE_WORDS/4] = { 0 };

void __attribute__((__interrupt__,no_auto_psv)) _DMA1Interrupt(void) {
	signed short* buf = DAC_CurrentBuffer ? TxBufferB : TxBufferA;
	signed short* src = DACBuffer+DACBuffer_readpos;
	IFS0bits.DMA1IF = 0;
	DAC_CurrentBuffer ^= 1;

	if( DAC_Paused ) {
		memset(buf, 0, sizeof(TxBufferA));
	} else {
		memcpy(buf, src, sizeof(TxBufferA));

		asm volatile("disi #12");
		DACBuffer_readpos += 256;
		if( DACBuffer_readpos == 256*DAC_BUFFER_PAGES )
			DACBuffer_readpos = 0;
		DACBuffer_full = 0;
		if( DACBuffer_readpos == DACBuffer_writepos )
			DACBuffer_empty = 1;
		asm volatile("disi #0");
	}
}

void dac_start_playback() {
	DMA1CONbits.CHEN = 1;
	DCICON1bits.DCIEN = 1;
	TXBUF0 = 0;
	LATBbits.LATB1 = 1;
}

void dac_stop_playback() {
	DMA1CONbits.CHEN = 0;
	DCICON1bits.DCIEN = 0;
	LATBbits.LATB1 = 0;
}

extern void process_dac_data(signed short* data);
void dac_write(signed short* data, unsigned char num_channels, unsigned short volume) {
	unsigned int i, j;
	signed short* dest = DACBuffer+DACBuffer_writepos;

	if( num_channels == 1 ) {
		for( j = 0; j < 4; ++j ) {
			while( DACBuffer_full )
				Nop();
			for( i = 0; i < 64; ++i ) {
				signed long sample  = (signed long)data[0] * volume;
				dest[0] = sample<<4;
				dest[1] = sample>>8;
				dest[2] = sample<<4;
				dest[3] = sample>>8;
				dest += 4;
				data += 1;
			}
			asm volatile("disi #16");
			DACBuffer_writepos += 256;
			if( DACBuffer_writepos == 256*DAC_BUFFER_PAGES )
				DACBuffer_writepos = 0;
			if( DACBuffer_writepos == DACBuffer_readpos )
				DACBuffer_full = 1;
			DACBuffer_empty = 0;
			asm volatile("disi #0");
		}
	} else {
		process_dac_data(data);
		for( j = 0; j < 2; ++j ) {
			while( DACBuffer_full )
				Nop();
			for( i = 0; i < 128; ++i ) {
				signed long sample  = (signed long)data[0] * volume;
				dest[0] = sample<<4;
				dest[1] = sample>>8;
				dest += 2;
				data += 1;
			}
			asm volatile("disi #16");
			DACBuffer_writepos += 256;
			if( DACBuffer_writepos == 256*DAC_BUFFER_PAGES )
				DACBuffer_writepos = 0;
			if( DACBuffer_writepos == DACBuffer_readpos )
				DACBuffer_full = 1;
			DACBuffer_empty = 0;
			asm volatile("disi #0");
		}
	}
}

void dac_pause(bool Paused) {
  DAC_Paused = Paused;
}

void dac_clear_buffer() {
  memset(DACBuffer, 0, 512*DAC_BUFFER_PAGES);
  DACBuffer_writepos = DACBuffer_readpos = 0;
  DACBuffer_full = 0;
  DACBuffer_empty = 1;
}

void dac_reset_buffer() {
  DACBuffer_writepos = DACBuffer_readpos = 0;
  DACBuffer_full = 0;
  DACBuffer_empty = 1;
}

unsigned char* dac_get_buffer() {
  return (unsigned char*)DACBuffer;
}

bool dac_is_running() {
  return DMA1CONbits.CHEN && !DAC_Paused;
}

unsigned char dac_get_free_buffers() {
  unsigned char full, empty;
  signed short readpos, writepos;

  asm volatile("disi #16");
  full = DACBuffer_full;
  empty = DACBuffer_empty;
  readpos = DACBuffer_readpos;
  writepos = DACBuffer_writepos;
  asm volatile("disi #0");

  if( full )
    return 0;
  else if( empty )
    return DAC_BUFFER_PAGES;

  writepos -= readpos;
  if( writepos < 0 )
    writepos += 256 * DAC_BUFFER_PAGES;
  return writepos >> 8;
}
